from django.db import models
from django.utils import timezone
from utilz.models import ProductsModel, SachetModel


SHIFT_CHOICE = (("MORNING", "MORNING"), ("AFTERNOON", "AFTERNOON"), ("NIGHT", "NIGHT"), ("DAY", "DAY"))
STATUS_CHOICE = (('CREATED', 'CREATED'), ('ANALYSED', 'ANALYSED'), ('CHECKED', 'CHECKED'), ('APPROVED', 'APPROVED'))

class SachetInspectionModel(models.Model):
    mc_id = models.ForeignKey(SachetModel, related_name='m_id', on_delete=models.PROTECT)
    date = models.DateField(default=timezone.now)
    shift = models.CharField(max_length=12, choices = SHIFT_CHOICE)
    product_id = models.ForeignKey(ProductsModel, related_name='product_si', on_delete=models.PROTECT)
    blend_number = models.CharField(max_length=24)
    inspector = models.CharField(max_length=30)
    supervisor = models.CharField(max_length=30, null=True, blank=True)
    foreman = models.CharField(max_length=30)
    machine_operators = models.CharField(max_length=30, null=True, blank=True)
    packers = models.CharField(max_length=60, null=True, blank=True)
    status = models.CharField(max_length=12, choices = STATUS_CHOICE, default='CREATED')

    comment = models.TextField(null=True, blank=True)

    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         verbose_name_plural = 'SACHET INSPECTION SHIFT FORM'
         ordering = ('date', 'shift', 'mc_id__machine')


    def __str__(self) -> str:
        return f"DATE : {self.date}, MACHINE : {self.mc_id.machine}, SHIFT : {self.shift}, PRODUCT : {self.product_id.product_name_short}"


MATERIAL_CHOICE = (('FILM_CODE', 'FILM_CODE'), ('CASE_CODE', 'CASE_CODE'), ('MINIPACK_CODE', 'MINIPACK_CODE'), ('POUCH_CODE', 'POUCH_CODE'))
class SachetInspectionPkgCodeModel(models.Model):
    machine_id = models.ForeignKey(SachetInspectionModel, related_name='pkg_code', on_delete=models.CASCADE)
    material_type = models.CharField(max_length=24, choices= MATERIAL_CHOICE)
    pkg_code = models.CharField(max_length=30)
    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         verbose_name_plural = 'SACHET INSPECTION PKG CODES'
         ordering = ('machine_id',)


    def __str__(self) -> str:
        return f"MACHINE : {self.machine_id.mc_id.machine}"




HOUR_CHOICE = (('00','00'),('01','01'),('02','02'),('03','03'),('04','04'),('05','05'),('06','06'),('07','07'),('08','08'),('09','09'),('10','10'),('11','11'),('12','12'),('13','13'),('14','14'),('15','15'),('16','16'),('17','17'),('18','18'),('19','19'),('20','20'),('21','21'),('22','22'),('23','23'))
class SachetInspectionFaultModel(models.Model):
    machine_id = models.ForeignKey(SachetInspectionModel, related_name='fault', on_delete=models.CASCADE)
    hour = models.CharField(max_length=6, choices= HOUR_CHOICE)
    fault = models.TextField()
    corrective_action = models.TextField(null=True, blank=True)
    result = models.TextField(null=True, blank=True)
    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         verbose_name_plural = 'SACHET INSPECTION FAULT BOX'
         ordering = ('machine_id__date', 'machine_id__shift', 'machine_id__mc_id__machine', 'hour')

    def __str__(self) -> str:
        return f"DATE : {self.machine_id.date}, MACHINE : {self.machine_id.mc_id.machine}, SHIFT : {self.machine_id.shift}"


from utilz.models import SachetDefectsModel
CAUSE_CHOICES = (("RAW MATERIAL", "M : RAW MATERIAL"), ("ENGINEERING", "E : ENGINEERING"), ("MAINTENANCE SETTING", "W : MAINTENANCE SETTING"), ("HUMAN", "O : HUMAN"))
class SachetInspectionDefectModel(models.Model):
    machine_id = models.ForeignKey(SachetInspectionModel, related_name='defect', on_delete=models.CASCADE)
    hour = models.CharField(max_length=6, choices= HOUR_CHOICE)
    defect_id = models.ForeignKey(SachetDefectsModel, related_name='defect_id', on_delete=models.CASCADE)
    quantity = models.IntegerField()
    defect_cause = models.CharField(max_length=24, choices = CAUSE_CHOICES)
    comment = models.TextField(null=True, blank=True)
    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         verbose_name_plural = 'SACHET INSPECTION DEFECT RECORD'
         ordering = ('machine_id__date', 'machine_id__shift', 'machine_id__mc_id__machine', 'hour')

    def __str__(self) -> str:
        return f"DATE : {self.machine_id.date}, MACHINE : {self.machine_id.mc_id.machine}, SHIFT : {self.machine_id.shift}"



class SachetInspectionDespatchPalletModel(models.Model):
    machine_id = models.ForeignKey(SachetInspectionModel, related_name='sachet_pallet', on_delete=models.CASCADE)
    qa_pallet_code = models.CharField(max_length=30)
    production_batch_code = models.CharField(max_length=30, null=True, blank=True)
    production_pallet_code = models.CharField(max_length=30, null=True, blank=True)
    case_code = models.CharField(max_length=30)
    comment = models.TextField(null=True, blank=True)
    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         verbose_name_plural = 'SACHET INSPECTION DESPATCHED PALLET RECORD'
         ordering = ('machine_id__date', 'machine_id__shift', 'machine_id__mc_id__machine', 'qa_pallet_code')

    def __str__(self) -> str:
        return f"DATE : {self.machine_id.date}, SACHET : {self.machine_id.mc_id.machine}, SHIFT : {self.machine_id.shift}"



class SachetInspectionWorkFlowModel(models.Model):
    machine_id = models.OneToOneField(SachetInspectionModel, related_name='workflow', on_delete=models.CASCADE)

    analysed_by = models.CharField(max_length=24, null=True, blank=True)
    analysed_comment = models.TextField(null=True, blank=True)
    analysed_datetime = models.DateTimeField(null=True, blank=True)

    checked_by = models.CharField(max_length=24, null=True, blank=True)
    checked_comment = models.TextField(null=True, blank=True)
    checked_datetime = models.DateTimeField(null=True, blank=True)

    approved_by = models.CharField(max_length=24, null=True, blank=True)
    approved_comment = models.TextField(null=True, blank=True)
    approved_datetime = models.DateTimeField(null=True, blank=True)

    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         verbose_name_plural = 'SACHET INSPECTION SHIFT WORKFLOW'
         ordering = ('machine_id__date', 'machine_id__shift', 'machine_id__mc_id__machine')

    def __str__(self):
        return f'DATE : {self.machine_id.date}, MACHINE : {self.machine_id.mc_id.machine}, SHIFT : {self.machine_id.shift}'




from django.core.validators import FileExtensionValidator
class SachetInspectionFilesModel(models.Model):
    machine_id = models.ForeignKey(SachetInspectionModel, related_name='files', on_delete=models.CASCADE)
    file = models.FileField(validators=[FileExtensionValidator(['png','jpg','jpeg','pdf','doc','docx','cdr'])], upload_to='files_folder', blank=True, null=True)
    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         verbose_name_plural = 'SACHET INSPECTION UPLOADED FILES'
         ordering = ('machine_id__date', 'machine_id__shift', 'machine_id__mc_id__machine')

    def __str__(self):
        return f'DATE : {self.machine_id.date}, LINE : {self.machine_id.mc_id.machine}, SHIFT : {self.machine_id.shift}'