from django.db import models
from django.contrib.auth.models import User

ROLES_CHOICES = (('INITIATOR', 'INITIATOR'), ('CHECKER' , 'CHECKER'), ('APPROVER', 'APPROVER'), ('SUPERADMIN', 'SUPERADMIN'))


class UserProfile(models.Model):
    user = models.OneToOneField(User, related_name='profile', on_delete=models.CASCADE)
    role = models.CharField(max_length=12, choices=ROLES_CHOICES)
    is_delegated = models.BooleanField(default=False)
    last_logout = models.DateTimeField(null=True, blank=True)
    created_by = models.CharField(max_length=55)
    datetime_created = models.DateTimeField(auto_now_add=True)

    class Meta:
         #verbose_name = 'LINE ANALYSIS'
         verbose_name_plural = 'USER PROFILE'
         ordering = ('role', 'user')

    def save(self, *args, **kwargs):
        self.first_name = self.user.first_name
        return super().save(*args, **kwargs)

    

    def __str__(self) -> str:
        return f"{self.user}"

### NB:
### USER = SAMPLER, ANALYZER
### SUPERUSER = CHECKER
### ADMIN = APPROVER
### SUPERADMIN = *

### has_been_delegated : gives the user role 1 step above
### can only been granted or removed by superior user role
### only applicable to




MODEL_CHOICES = (('LINE_ANALYSIS', 'LINE_ANALYSIS'), ('SACHET_ANALYSIS', 'SACHET_ANALYSIS'), ('LINE_INSPECTION', 'LINE_INSPECTION'), ('SACHET_INSPECTION', 'SACHET_INSPECTION'))
USER_ACTIVITY = (('CREATE', 'CREATE'), ('SUBMIT', 'SUBMIT'), ('CHECK','CHECK'), ('APPROVE','APPROVE'), ('UPDATE', 'UPDATE'), ('DELETE', 'DELETE'))

class UsersActivitiesModel(models.Model):
    model_name = models.CharField(max_length=24, choices=MODEL_CHOICES) 
    model_id = models.IntegerField()
    activity = models.CharField(max_length=12, choices=USER_ACTIVITY)
    description = models.CharField(max_length=30)
    
    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)

    user_ip = models.GenericIPAddressField()

    class Meta:
         verbose_name_plural = 'USER ACTIVITIES TRACKER'
         ordering = ('datetime_created',)

    def get_model_redirect_url(self):
        if self.model_name == 'LINE_ANALYSIS':
            section = 'LINE_ANALYSIS'
            redirect_url = f'/line/analysis/detail/{self.model_id}/'
        elif self.model_name == 'SACHET_ANALYSIS':
            section = 'SACHET_ANALYSIS'
            redirect_url = f'/sachet/analysis/detail/{self.model_id}/'
        elif self.model_name == 'LINE_INSPECTION':
            section = 'LINE_INSPECTION'
            redirect_url = f'/line/inspection/detail/{self.model_id}/'
        elif self.model_name == 'SACHET_INSPECTION':
            section = 'SACHET_INSPECTION'
            redirect_url = f'/sachet/inspection/detail/{self.model_id}/'
        else:
            section = 'NA'
            redirect_url = '/'
        return (redirect_url, section)


    def __str__(self):
        return f'{self.model_name}/ {self.activity}, USER : {self.created_by}'



class MessageModel(models.Model):
    sender = models.CharField(max_length=60)
    receiver = models.ForeignKey(User, related_name='mssg_receiver', on_delete=models.CASCADE)
    message = models.TextField()
    is_read = models.BooleanField(default=False)
    created_by = models.CharField(max_length=24)
    datetime_created = models.DateTimeField(auto_now_add=True)
    user_ip = models.GenericIPAddressField()

    class Meta:
         verbose_name_plural = 'CHAT MESSAGES'
         ordering = ('datetime_created',)

    def __str__(self):
        return f'SENDER {self.sender}, TO : {self.receiver}, DATE {self.datetime_created}'
